//-------------------------------------------------------------------------------------
// SID Monitor - Utility for Sudden Ionospheric Disturbances Monitoring Stations
// Copyright (C) 2006 - Lionel Loudet
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
//-------------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Text;
using System.Windows.Forms;
using System.IO;
using WeifenLuo.WinFormsUI;

namespace SID_monitor
{
    public partial class GOESDockablePanel : DockContent
    {

        public GOESDockablePanel()
        {
            InitializeComponent();

            // label GOES database Update
            updateLabelGOESUpdate();

        }


        #region Properties
        public bool GOES10Checked
        {
            get
            {
                return this.checkBoxGOES10.Checked;
            }
            set
            {
                this.checkBoxGOES10.Checked = value;
            }
        }

        public bool GOES11Checked
        {
            get
            {
                return this.checkBoxGOES11.Checked;
            }
            set
            {
                this.checkBoxGOES11.Checked = value;
            }
        }

        public bool GOES12Checked
        {
            get
            {
                return this.checkBoxGOES12.Checked;
            }
            set
            {
                this.checkBoxGOES12.Checked = value;
            }
        }

        public string GOESLastUpdateText
        {
            get
            {
                return this.labelLastGOESUpdateValue.Text;
            }
            set
            {
                this.labelLastGOESUpdateValue.Text = value;
            }
        }

        #endregion

        #region Checkboxes Methods
        private void checkBoxGOES10_CheckedChanged(object sender, EventArgs e)
        {
            OnGOES10ShowChanged(new GOESShowChangeEventArgs(checkBoxGOES10.Checked));
        }

        private void checkBoxGOES11_CheckedChanged(object sender, EventArgs e)
        {
            OnGOES11ShowChanged(new GOESShowChangeEventArgs(checkBoxGOES11.Checked));
        }

        private void checkBoxGOES12_CheckedChanged(object sender, EventArgs e)
        {
            OnGOES12ShowChanged(new GOESShowChangeEventArgs(checkBoxGOES12.Checked));
        }


        #endregion

        #region GOES Pane Methods
        /// <summary>
        /// An immediate refresh of the GOES database is asked
        /// </summary>
        private void buttonUpdateGOES_Click(object sender, EventArgs e)
        {
            OnGOESDownloadRequested(e);
        }

        /// <summary>
        /// Updates the label on the GOES pane that indicates the last GOES database update time
        /// </summary>
        public void updateLabelGOESUpdate()
        {
            if (!File.Exists(SID_monitor.Properties.Settings.Default.RRDToolGOES))
            {
                this.labelLastGOESUpdateValue.Text = "          N/A";
            }
            else
            {
                SIDMonitorDateTime lastGOESDatabaseUpdateTime;
                try
                {
                    lastGOESDatabaseUpdateTime = RRDTool.getLastDatabaseUpdateTime(SID_monitor.Properties.Settings.Default.RRDToolGOES, SID_monitor.Program.MainForm.outputTextBoxDockablePanel.Handle);
                }
                catch
                {
                    this.labelLastGOESUpdateValue.Text = "          N/A";
                    return;
                }
                this.labelLastGOESUpdateValue.Text = "          " + lastGOESDatabaseUpdateTime.WinTime.ToString("dd MMM yyyy HH:mm:ss") + " UTC";
            }
        }
        #endregion

        #region Events

        /// <summary>
        /// GOES10ShowChanged, GOES11ShowChanged and GOES12ShowChanged Event
        /// </summary>

        public class GOESShowChangeEventArgs : EventArgs
        {
            private bool show = false;

            public GOESShowChangeEventArgs(bool show)
            {
                this.show = show;
            }

            public bool Show
            {
                get
                {
                    return Show;
                }
            }
        }

        public delegate void GOESShowChangeEventHandler(object sender, GOESShowChangeEventArgs e);

        public event GOESShowChangeEventHandler GOES10ShowChanged; // the GOES-10 show checkbox has changed
        public event GOESShowChangeEventHandler GOES11ShowChanged; // the GOES-11 show checkbox has changed
        public event GOESShowChangeEventHandler GOES12ShowChanged; // the GOES-12 show checkbox has changed

        protected virtual void OnGOES10ShowChanged(GOESShowChangeEventArgs e)
        {
            GOESShowChangeEventHandler handler = GOES10ShowChanged;
            if (handler != null)
            {
                // Invokes the delegates. 
                handler(this, e);
            }
        }

        protected virtual void OnGOES11ShowChanged(GOESShowChangeEventArgs e)
        {
            GOESShowChangeEventHandler handler = GOES11ShowChanged;
            if (handler != null)
            {
                // Invokes the delegates. 
                handler(this, e);
            }
        }

        protected virtual void OnGOES12ShowChanged(GOESShowChangeEventArgs e)
        {
            GOESShowChangeEventHandler handler = GOES12ShowChanged;
            if (handler != null)
            {
                // Invokes the delegates. 
                handler(this, e);
            }
        }


        /// <summary>
        /// GOESDownloadRequested Event
        /// </summary>

        public event EventHandler GOESDownloadRequested;  // GOES download is requested

        protected virtual void OnGOESDownloadRequested(EventArgs e)
        {
            EventHandler handler = GOESDownloadRequested;
            if (handler != null)
            {
                // Invokes the delegates. 
                handler(this, e);
            }
        }
        #endregion

     
        

    }
}
